#include "TerrainManager.h"
#include <io.h>

TerrainManager :: TerrainManager() {
   tmesh = NULL;
   surfaceEq = NULL;
   texscale = 0.3f;
   curr_map_file = -1;
   
   float bc[] = {0.2f,1,0.2f};
   float mc[] = {0.8f,0,0};
   setTerrainColor(bc,mc);

   float up[] = {0.2f,0.4f,0.6f};
   float down[] = {0.4f,0.8f,1.0f};
   setSkyColor(up,down);

}

TerrainManager :: ~TerrainManager() {
   if (tmesh)
      delete tmesh;
   if (surfaceEq)
      delete [] surfaceEq;
}

TerrainManager* TerrainManager :: instance() {
   static TerrainManager *tman = new TerrainManager;
   return tman;
}

void TerrainManager :: setTerrainColor(float basecol[3],float modcol[3]) {
   memcpy(this->basecol,basecol,sizeof(float)*3);
   memcpy(this->modcol,modcol,sizeof(float)*3);
}

void TerrainManager :: setSkyColor(float colup[3],float coldown[3]) {
   memcpy(this->skycolup,colup,sizeof(float)*3);
   memcpy(this->skycoldown,coldown,sizeof(float)*3);
}

void TerrainManager :: initWithMesh(tMesh *mesh) {
   if (tmesh)
      delete tmesh;
   if (surfaceEq)
      delete [] surfaceEq;
}

void TerrainManager :: initFromDirectory() {

   struct _finddata_t c_file;
   long hFile;

   /* Find first .map file in data/ directory */
   if( (hFile = _findfirst( "data/*.map", &c_file )) == -1L ) {
      printf( "No *.map files in data/ directory!\n" );
   }
   else {
      
      mapFiles.push_back(string("data/")+string(c_file.name));
      curr_map_file = 0;

      /* Find the rest of the .map files */
      while( _findnext( hFile, &c_file ) == 0 ) {
         mapFiles.push_back(string("data/")+string(c_file.name));
      }

      _findclose( hFile );
   }

   initNextMapFromDirectory();
   
}

void TerrainManager :: initNextMapFromDirectory() {

   if (curr_map_file == -1)
      return;

   if (curr_map_file < mapFiles.size()-1)
      ++curr_map_file;      
   else
      curr_map_file = 0;

   initFromFile(mapFiles[curr_map_file].c_str());

}

const char* TerrainManager ::getCurrentMapFile() {

   static char *none = "none";
   
   if (curr_map_file == -1)
      return none;
   else
      return mapFiles[curr_map_file].c_str();

}

void TerrainManager :: initFromFile(const char *filename) {   
  
   PropReader pr(filename);
   if (pr.getError() != 0)
      return;

   if (tmesh)
      delete tmesh;
   if (surfaceEq)
      delete [] surfaceEq;

   decorations.clear();

   pr.getString("name",mapname);
   pr.getString("description",mapdesc);
   char texname[256];
   pr.getString("texture",texname);
   char deconame[256];
   pr.getString("decorations",deconame);
   skycolup[0] = pr.getFloat("sky_up.r");
   skycolup[1] = pr.getFloat("sky_up.g");
   skycolup[2] = pr.getFloat("sky_up.b");
   skycoldown[0] = pr.getFloat("sky_do.r");
   skycoldown[1] = pr.getFloat("sky_do.g");
   skycoldown[2] = pr.getFloat("sky_do.b");
   basecol[0] = pr.getFloat("terrbase_col.r");
   basecol[1] = pr.getFloat("terrbase_col.g");
   basecol[2] = pr.getFloat("terrbase_col.b");
   modcol[0] = pr.getFloat("terrmod_col.r");
   modcol[1] = pr.getFloat("terrmod_col.g");
   modcol[2] = pr.getFloat("terrmod_col.b");
   texscale = pr.getFloat("detail_scale");
   float addy = pr.getFloat("addy");

   ttex = new Texture(texname);
   ttex->build();

   if (strlen(deconame) > 0)
      loadDecorations(deconame);

   const int col_tris = 20;
   float heights[col_tris+1];
      
   for (int i=0;i<col_tris+1;++i) {
      char hstr[256];
      sprintf(hstr,"h.%d",i);
      heights[i] = pr.getFloat(hstr) + addy;
   }

   tmesh = new tMesh;
   tmesh->faceCount = col_tris * 2;
   tmesh->vertexCount = tmesh->faceCount * 3;
   tmesh->uvCount = tmesh->vertexCount;
   tmesh->normalCount = 1;
   tmesh->vertices = new tVertex[tmesh->vertexCount];
   tmesh->faces = new tFace[tmesh->faceCount];
   tmesh->uvs = new tUV[tmesh->uvCount];
   tmesh->normals = new Vector3d(0,0,0);
   surfaceEqCnt = col_tris;
   surfaceEq = new tLineEq[surfaceEqCnt];

   terrain_minx = 0; terrain_maxx = 20.0f;
   terrain_miny = 0; terrain_maxy = 20.0f;

   for (i=0;i<col_tris;++i) {

      int f = i * 2;
      
      tVertex *vptr = &tmesh->vertices[f*3];
      tUV *uvptr = &tmesh->uvs[f*3];
      tFace *fptr = &tmesh->faces[f];

      fptr->v[0] = fptr->uv[0] = f * 3;
      fptr->v[1] = fptr->uv[1] = f * 3 + 1;
      fptr->v[2] = fptr->uv[2] = f * 3 + 2;
      vptr[0].x = i;
      vptr[0].y = heights[i];
      vptr[0].z = 0;
      vptr[1].x = i;
      vptr[1].y = 0;
      vptr[1].z = 0;
      vptr[2].x = i + 1;
      vptr[2].y = heights[i+1];
      vptr[2].z = 0;
      uvptr[0].u = vptr[0].x;
      uvptr[0].v = vptr[0].y;
      uvptr[1].u = vptr[1].x;
      uvptr[1].v = vptr[1].y;
      uvptr[2].u = vptr[2].x;
      uvptr[2].v = vptr[2].y;
      fptr->n[0] = 0;
      fptr->n[1] = 0;
      fptr->n[2] = 0;

      surfaceEq[i].m = (vptr[0].y - vptr[2].y) / (vptr[0].x - vptr[2].x);
      surfaceEq[i].n = vptr[0].y - surfaceEq[i].m * vptr[0].x;
      
      ++f;

      vptr += 3;
      uvptr += 3;
      ++fptr;

      fptr->v[0] = fptr->uv[0] = f * 3;
      fptr->v[1] = fptr->uv[1] = f * 3 + 1;
      fptr->v[2] = fptr->uv[2] = f * 3 + 2;
      vptr[0].x = i + 1;
      vptr[0].y = heights[i+1];
      vptr[0].z = 0;
      vptr[1].x = i;
      vptr[1].y = 0;
      vptr[1].z = 0;
      vptr[2].x = i + 1;
      vptr[2].y = 0;
      vptr[2].z = 0;
      uvptr[0].u = vptr[0].x;
      uvptr[0].v = vptr[0].y;
      uvptr[1].u = vptr[1].x;
      uvptr[1].v = vptr[1].y;
      uvptr[2].u = vptr[2].x;
      uvptr[2].v = vptr[2].y;
      fptr->n[0] = 0;
      fptr->n[1] = 0;
      fptr->n[2] = 0;
   
   }

   //
   for (i=0;i<HMAP_SIZE;++i) {
      hmap[i] = heights[i];   
   }
   
   
}

void TerrainManager :: initRandom(float minx,float maxx,float miny,float maxy,float hdiff,int lod) {

   if (tmesh)
      delete tmesh;
   if (surfaceEq)
      delete [] surfaceEq;
   
   ttex = new Texture("data/detail.jpg");
   ttex->build();
   
   strcpy(mapname,"random_map");
   strcpy(mapdesc,"random_map_desc");

   tmesh = new tMesh;
   // final addition for border mesh
   int col_tris = pow(2,lod);
   tmesh->faceCount = col_tris * 2;
   // final addition for border mesh
   tmesh->vertexCount = tmesh->faceCount * 3;
   tmesh->uvCount = tmesh->vertexCount;
   tmesh->normalCount = 1;
   tmesh->vertices = new tVertex[tmesh->vertexCount];
   tmesh->faces = new tFace[tmesh->faceCount];
   tmesh->uvs = new tUV[tmesh->uvCount];
   tmesh->normals = new Vector3d(0,0,0);
   surfaceEqCnt = col_tris;
   surfaceEq = new tLineEq[surfaceEqCnt];
   
   // create random terrain mesh by using midpoint displacement algorithm

   curr_face_idx = 0;
   base_vertex_displacement = 5;
   height_diff = hdiff;
   this->lod = lod;
   terrain_minx = minx;
   terrain_maxx = maxx;
   terrain_miny = miny;
   terrain_maxy = maxy;
   initRandom_recursive(minx,maxx,miny,maxy,lod);

}
   
void TerrainManager :: initRandom_recursive(float minx,float maxx,float miny,float maxy,int lod) {

   if (lod == 0) {
   
      tVertex *vptr = &tmesh->vertices[curr_face_idx*3];
      tUV *uvptr = &tmesh->uvs[curr_face_idx*3];
      tFace *fptr = &tmesh->faces[curr_face_idx];

      //if (miny < terrain_miny) miny = terrain_miny;
      //if (maxy > terrain_maxy) maxy = terrain_maxy;

      fptr->v[0] = fptr->uv[0] = curr_face_idx * 3;
      fptr->v[1] = fptr->uv[1] = curr_face_idx * 3 + 1;
      fptr->v[2] = fptr->uv[2] = curr_face_idx * 3 + 2;
      vptr[0].x = minx;
      vptr[0].y = miny;
      vptr[0].z = 0;
      vptr[1].x = minx;
      vptr[1].y = 0;
      vptr[1].z = 0;
      vptr[2].x = maxx;
      vptr[2].y = maxy;
      vptr[2].z = 0;
      uvptr[0].u = vptr[0].x;
      uvptr[0].v = vptr[0].y;
      uvptr[1].u = vptr[1].x;
      uvptr[1].v = vptr[1].y;
      uvptr[2].u = vptr[2].x;
      uvptr[2].v = vptr[2].y;
      fptr->n[0] = 0;
      fptr->n[1] = 0;
      fptr->n[2] = 0;

      surfaceEq[curr_face_idx/2].m = (vptr[0].y - vptr[2].y) / (vptr[0].x - vptr[2].x);
      surfaceEq[curr_face_idx/2].n = vptr[0].y - surfaceEq[curr_face_idx/2].m * vptr[0].x;
      
      ++curr_face_idx;

      vptr += 3;
      uvptr += 3;
      ++fptr;

      fptr->v[0] = fptr->uv[0] = curr_face_idx * 3;
      fptr->v[1] = fptr->uv[1] = curr_face_idx * 3 + 1;
      fptr->v[2] = fptr->uv[2] = curr_face_idx * 3 + 2;
      vptr[0].x = maxx;
      vptr[0].y = maxy;
      vptr[0].z = 0;
      vptr[1].x = minx;
      vptr[1].y = 0;
      vptr[1].z = 0;
      vptr[2].x = maxx;
      vptr[2].y = 0;
      vptr[2].z = 0;
      uvptr[0].u = vptr[0].x;
      uvptr[0].v = vptr[0].y;
      uvptr[1].u = vptr[1].x;
      uvptr[1].v = vptr[1].y;
      uvptr[2].u = vptr[2].x;
      uvptr[2].v = vptr[2].y;
      fptr->n[0] = 0;
      fptr->n[1] = 0;
      fptr->n[2] = 0;

      ++curr_face_idx;

      return;
   }
      

   float midx = minx + (maxx - minx) / 2;
   float midy = miny + (maxy - miny) / 2;

   static float base_height_diff = height_diff;
   if (lod == this->lod-1)
      height_diff = base_height_diff;
   
   midy += height_diff * (rand()%100)*0.01f * ((rand()%2)?1:-1);
   height_diff *= 0.60f;

   // left part
   initRandom_recursive(minx,midx,miny,midy,lod-1);
   
   // right part
   initRandom_recursive(midx,maxx,midy,maxy,lod-1);

}

void TerrainManager :: loadDecorations(const char *filename) {

   FILE *fp = fopen(filename,"rt");
   if (!fp)
      return;

   // file format
   // type x y sx sy rz

   while (!feof(fp)) {
   
      char line[256];
      fgets(line,255,fp);
      
      if (strlen(line) < 2 || line[0] == '#' || line[0] == '/')
         continue;
      
      tDecoration d;
      char texname[256];
      
      char *token = strtok(line," \t\n");
      strcpy(texname,token);
      d.type = atoi(strtok(NULL," \t\n"));
      d.x = atof(strtok(NULL," \t\n"));
      d.y = atof(strtok(NULL," \t\n"));
      d.sx = atof(strtok(NULL," \t\n"));
      d.sy = atof(strtok(NULL," \t\n"));
      d.rz = atof(strtok(NULL," \t\n"));

      char *src = strtok(NULL," \t\n");
      char *dst = strtok(NULL," \t\n");
      int srcm=GL_ONE,dstm=GL_ONE;
      if (strcmp(src,"GL_SRC_ALPHA") == 0)           srcm = GL_SRC_ALPHA;
      if (strcmp(src,"GL_ONE_MINUS_SRC_ALPHA") == 0) srcm = GL_ONE_MINUS_SRC_ALPHA;
      if (strcmp(src,"GL_DST_COLOR") == 0)           srcm = GL_DST_COLOR;
      if (strcmp(src,"GL_ONE_MINUS_DST_COLOR") == 0) srcm = GL_ONE_MINUS_DST_COLOR;
      if (strcmp(src,"GL_DST_ALPHA") == 0)           srcm = GL_DST_ALPHA;
      if (strcmp(src,"GL_ONE_MINUS_DST_ALPHA") == 0) srcm = GL_ONE_MINUS_DST_ALPHA;
      if (strcmp(dst,"GL_SRC_COLOR") == 0)           dstm = GL_SRC_COLOR;
      if (strcmp(dst,"GL_ONE_MINUS_SRC_COLOR") == 0) dstm = GL_ONE_MINUS_SRC_COLOR;
      if (strcmp(dst,"GL_SRC_ALPHA") == 0)           dstm = GL_SRC_ALPHA;
      if (strcmp(dst,"GL_ONE_MINUS_SRC_ALPHA") == 0) dstm = GL_ONE_MINUS_SRC_ALPHA;
      if (strcmp(dst,"GL_DST_ALPHA") == 0)           dstm = GL_DST_ALPHA;
      if (strcmp(dst,"GL_ONE_MINUS_DST_ALPHA") == 0) dstm = GL_ONE_MINUS_DST_ALPHA;
      d.src_blendmode = srcm;
      d.dst_blendmode = dstm;
      
      d.tex = new Texture(texname);
      d.tex->build();
      decorations.push_back(d);
   
   }


   fclose(fp);

}

void TerrainManager :: saveMapToFile(const char *filename) {

   FILE *fp = fopen(filename,"wt");

   fprintf(fp,"// saved map file\n");
   fprintf(fp,"name=%s\ndescription=%s\ntexture=%s\ndetail_scale=%f\n",
      mapname,mapdesc,ttex->getPath(),texscale);
   fprintf(fp,"sky_up.r=%f\nsky_up.g=%f\nsky_up.b=%f\n",skycolup[0],skycolup[1],skycolup[2]);
   fprintf(fp,"sky_do.r=%f\nsky_do.g=%f\nsky_do.b=%f\n",skycoldown[0],skycoldown[1],skycoldown[2]);
   fprintf(fp,"terrbase_col.r=%f\nterrbase_col.g=%f\nterrbase_col.b=%f\n",basecol[0],basecol[1],basecol[2]);
   fprintf(fp,"terrmod_col.r=%f\nterrmod_col.g=%f\nterrmod_col.b=%f\n",modcol[0],modcol[1],modcol[2]);

   
   for (int i=0;i<HMAP_SIZE;++i) {
      fprintf(fp,"h.%d=%f\n",i,hmap[i]);
   }

   fclose(fp);
}

void TerrainManager :: render() {

   renderBackground();
   
   glColorMask(0,0,0,0);
   glEnable(GL_STENCIL_TEST);
   glStencilFunc(GL_ALWAYS,1,0xffffffff);
   glStencilOp(GL_REPLACE, GL_REPLACE, GL_REPLACE);
   renderMap();
   glDisable(GL_STENCIL_TEST);     
   glColorMask(1,1,1,1);
   //glStencilFunc(GL_EQUAL, 1, 0xffffffff);
   //glStencilOp(GL_KEEP, GL_KEEP, GL_KEEP);	
   
   ttex->begin();
   glMatrixMode(GL_TEXTURE);
   glPushMatrix();
   glScalef(texscale,texscale,1);
   
   glBegin(GL_TRIANGLE_STRIP);
   glTexCoord2f(0,hmap[0]);
   glColor4f(basecol[0]+hmap[0]*modcol[0],basecol[1]+hmap[0]*modcol[1],basecol[2]+hmap[0]*modcol[2],1.0f);
   glVertex2f(0,hmap[0]);
   glTexCoord2f(0,0);
   glColor4f(basecol[0],basecol[1],basecol[2],1.0f);
   glVertex2f(0,0);
   glTexCoord2f(1,hmap[1]);
   glColor4f(basecol[0]+hmap[1]*modcol[0],basecol[1]+hmap[1]*modcol[1],basecol[2]+hmap[1]*modcol[2],1.0f);
   glVertex2f(1,hmap[1]);
   for (int i=0;i<HMAP_SIZE-1;++i) {
      glTexCoord2f(i,0);
      glColor4f(basecol[0],basecol[1],basecol[2],1.0f);    
      glVertex2f(i,0);
      glTexCoord2f(i+1,hmap[i+1]);
      glColor4f(basecol[0]+hmap[i+1]*modcol[0],basecol[1]+hmap[i+1]*modcol[1],basecol[2]+hmap[i+1]*modcol[2],1.0f);    
      glVertex2f(i+1,hmap[i+1]);
   }
   glTexCoord2f(i,0);
   glColor4f(basecol[0],basecol[1],basecol[2],1.0f);
   glVertex2f(i,0);
   glEnd();

   glPopMatrix();
   glMatrixMode(GL_MODELVIEW);
   
   ttex->end();

   glEnable(GL_BLEND);
   //glBlendFunc(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA);
   //glBlendFunc(GL_ONE,GL_ONE);
   //glEnable(GL_ALPHA_TEST);
   //glAlphaFunc(GL_GREATER, 0.0f);

   glColor3f(1,1,1);
   DecorationList::iterator di = decorations.begin();
   while (di != decorations.end()) {
   
      tDecoration &d = *di;
      glPushMatrix();
      glTranslatef(d.x,d.y,0);
      glRotatef(d.rz,0,0,1);
      glScalef(d.sx,d.sy,1);
      glBlendFunc(d.src_blendmode,d.dst_blendmode);
      d.tex->begin();
      glBegin(GL_QUADS);
      glTexCoord2f(0,0); glVertex2f(0,0);
      glTexCoord2f(1,0); glVertex2f(1,0);
      glTexCoord2f(1,1); glVertex2f(1,1);
      glTexCoord2f(0,1); glVertex2f(0,1);
      glEnd();
      d.tex->end();
      glPopMatrix();
   
      ++di;
   }
   glDisable(GL_BLEND);


}

void TerrainManager :: renderMap() {

   glDisable(GL_TEXTURE_2D);
   glDisable(GL_LIGHTING);      
   glBegin(GL_TRIANGLE_STRIP);
   glVertex2f(0,hmap[0]);
   glVertex2f(0,0);
   glVertex2f(1,hmap[1]);
   for (int i=0;i<HMAP_SIZE-1;++i) {
      glVertex2f(i,0);
      glVertex2f(i+1,hmap[i+1]);
   }
   glVertex2f(i,0);
   glEnd();

}

void TerrainManager :: renderBackground() {

   glDisable(GL_BLEND);
   glDisable(GL_TEXTURE_2D);
   glPushMatrix();
   glLoadIdentity();
   glBegin(GL_QUADS);
   glColor3fv(skycolup);
   glVertex2f(terrain_maxx,terrain_maxy);
   glVertex2f(0,terrain_maxy);
   glColor3fv(skycoldown);
   glVertex2f(0,0);
   glVertex2f(terrain_maxx,0);
   glEnd();
   glPopMatrix();

}

float TerrainManager :: getHot(float x) {

   /*
   float surface_w = (terrain_maxx - terrain_minx) / (tmesh->faceCount/2);
   int curr_surface = (x - terrain_minx ) / surface_w;

   if (curr_surface < surfaceEqCnt)
      return surfaceEq[curr_surface].m * x + surfaceEq[curr_surface].n;

   */

   if (x >= 0 && x < HMAP_SIZE) {
   
      float x1 = (int) x;
      float y1 = hmap[(int)x1];
      float x2 = x1 + 1;
      float y2 = hmap[(int)x2];

      //(x - x1) / (x2 - x1) = (hot - y1) / (y2 - y1)
      return ((x - x1) * (y2 - y1) / (x2 - x1)) + y1;   
   }

   return -1;
}

bool TerrainManager :: testCollision(float x,float y) {

   if (getHot(x) >= y)
      return true;
   else
      return false;

}

void TerrainManager :: deform(float x) {
   if (x >= terrain_minx && x <= terrain_maxx) {
      /*
      for (int i=0;i<surfaceEqCnt;++i) {
         //int vidx[3] = {};
      }
      */

      float surface_w = (terrain_maxx - terrain_minx) / (tmesh->faceCount/2);
      int tri = (x - terrain_minx ) / surface_w;

      const float d = 0.1f;

      tFace *fptr;
      if (tri-2 >0) {
         fptr = &tmesh->faces[tri-2];
         tmesh->vertices[fptr->v[2]].y -= d;
         fptr = &tmesh->faces[tri-1];
         tmesh->vertices[fptr->v[0]].y -= d;
      }
      else
         fptr = &tmesh->faces[tri];

      ++fptr;
      tmesh->vertices[fptr->v[0]].y -= d;
      tmesh->vertices[fptr->v[2]].y -= d;
      ++fptr;
      tmesh->vertices[fptr->v[0]].y -= d;
      tmesh->vertices[fptr->v[2]].y -= d;
      ++fptr;
      tmesh->vertices[fptr->v[0]].y -= d;


   }
}


